/*      SCCS Id: @(#)invent.c   3.1     93/05/17        */
/* Copyright (c) Stichting Mathematisch Centrum, Amsterdam, 1985. */
/* NetHack may be freely redistributed.  See license for details. */

#include "hack.h"
#include "artifact.h"

#define NOINVSYM        '#'
#define CONTAINED_SYM   '>'     /* designator for inside a container */

#ifdef OVL1
static void NDECL(reorder_invent);
static boolean FDECL(mergable,(struct obj *,struct obj *));
static int FDECL(merged,(struct obj *,struct obj *,int));
#endif /* OVL1 */
STATIC_DCL void FDECL(assigninvlet,(struct obj *));
STATIC_DCL void FDECL(unlinkinv,(struct obj*));
STATIC_DCL void FDECL(compactify,(char *));
STATIC_PTR int FDECL(ckunpaid,(struct obj *));
#ifdef OVLB
static struct obj *FDECL(find_unpaid,(struct obj *,struct obj **));
static boolean NDECL(wearing_armor);
static boolean FDECL(is_worn,(struct obj *));
static boolean FDECL(is_fully_identified,(struct obj *));
#endif /* OVLB */
STATIC_DCL char FDECL(obj_to_let,(struct obj *));

#ifdef OVLB

static int lastinvnr = 51;      /* 0 ... 51 (never saved&restored) */

#ifdef WIZARD
/* wizards can wish for venom, which will become an invisible inventory
 * item without this.  putting it in inv_order would mean venom would
 * suddenly become a choice for all the inventory-class commands, which
 * would probably cause mass confusion.  the test for inventory venom
 * is only WIZARD and not wizard because the wizard can leave venom lying
 * around on a bones level for normal players to find.
 */
static char venom_inv[] = { VENOM_CLASS, 0 };   /* (constant) */
#endif

STATIC_OVL void
assigninvlet(otmp)
register struct obj *otmp;
{
	boolean inuse[52];
	register int i;
	register struct obj *obj;

	for(i = 0; i < 52; i++) inuse[i] = FALSE;
	for(obj = invent; obj; obj = obj->nobj) if(obj != otmp) {
		i = obj->invlet;
		if('a' <= i && i <= 'z') inuse[i - 'a'] = TRUE; else
		if('A' <= i && i <= 'Z') inuse[i - 'A' + 26] = TRUE;
		if(i == otmp->invlet) otmp->invlet = 0;
	}
	if((i = otmp->invlet) &&
	    (('a' <= i && i <= 'z') || ('A' <= i && i <= 'Z')))
		return;
	for(i = lastinvnr+1; i != lastinvnr; i++) {
		if(i == 52) { i = -1; continue; }
		if(!inuse[i]) break;
	}
	otmp->invlet = (inuse[i] ? NOINVSYM :
			(i < 26) ? ('a'+i) : ('A'+i-26));
	lastinvnr = i;
}

#endif /* OVLB */
#ifdef OVL1

/* note: assumes ASCII; toggling a bit puts lowercase in front of uppercase */
#define inv_rank(o) ((o)->invlet ^ 040)

/* sort the inventory; used by addinv() and doorganize() */
static void
reorder_invent()
{
	struct obj *otmp, *prev, *next;
	boolean need_more_sorting;

	do {
	    /*
	     * We expect at most one item to be out of order, so this
	     * isn't nearly as inefficient as it may first appear.
	     */
	    need_more_sorting = FALSE;
	    for (otmp = invent, prev = 0; otmp; ) {
		next = otmp->nobj;
		if (next && inv_rank(next) < inv_rank(otmp)) {
		    need_more_sorting = TRUE;
		    if (prev) prev->nobj = next;
		    else      invent = next;
		    otmp->nobj = next->nobj;
		    next->nobj = otmp;
		    prev = next;
		} else {
		    prev = otmp;
		    otmp = next;
		}
	    }
	} while (need_more_sorting);
}

#undef inv_rank

/* merge obj with otmp and delete obj if types agree */
STATIC_OVL int
merged(otmp, obj, lose)
register struct obj *otmp, *obj;
register int lose;
{
	if(mergable(otmp, obj)) {
		/* Approximate age: we do it this way because if we were to
		 * do it "accurately" (merge only when ages are identical)
		 * we'd wind up never merging any corpses.
		 * otmp->age = otmp->age*(1-proportion) + obj->age*proportion;
		 */
		otmp->age = ((otmp->age*otmp->quan) + (obj->age*obj->quan))
			/ (otmp->quan + obj->quan);
		otmp->quan += obj->quan;
		otmp->owt += obj->owt;
		if(!otmp->onamelth && obj->onamelth)
			otmp = oname(otmp, ONAME(obj), 1);
		if(lose) freeobj(obj);
		obfree(obj,otmp);       /* free(obj), bill->otmp */
		return(1);
	} else  return(0);
}

struct obj *
addinv(obj)
register struct obj *obj;
{
	register struct obj *otmp, *prev;

	
	if (obj->otyp == GOLD_PIECE) {
		u.ugold += obj->quan;
		flags.botl = 1;
		return obj;
	} else if (obj->otyp == AMULET_OF_YENDOR) {
		if (u.uhave.amulet) impossible ("already have amulet?");
		u.uhave.amulet = 1;
	} else if (obj->otyp == CANDELABRUM_OF_INVOCATION) {
		if (u.uhave.menorah) impossible ("already have candelabrum?");
		u.uhave.menorah = 1;
	} else if (obj->otyp == BELL_OF_OPENING) {
		if (u.uhave.bell) impossible ("already have silver bell?");
		u.uhave.bell = 1;
	} else if (obj->otyp == SPE_BOOK_OF_THE_DEAD) {
		if (u.uhave.book) impossible ("already have the book?");
		u.uhave.book = 1;
#ifdef MULDGN
	} else if (is_quest_artifact(obj)) {
		if (u.uhave.questart) impossible ("already have the artifact?");
		u.uhave.questart = 1;
		artitouch();
		set_artifact_intrinsic(obj, 1, W_ART);
#endif
	} else if(obj->oartifact) {
		set_artifact_intrinsic(obj, 1, W_ART);
	
	}
	
	/* merge if possible; find end of chain in the process */
	for (prev = 0, otmp = invent; otmp; prev = otmp, otmp = otmp->nobj)
	    if (merged(otmp, obj, 0)) {
		obj = otmp;
		goto added;
	    }
	/* didn't merge, so insert into chain */
	if (flags.invlet_constant || !prev) {
	    if (flags.invlet_constant) assigninvlet(obj);
	    obj->nobj = invent;         /* insert at beginning */
	    invent = obj;
	    if (flags.invlet_constant) reorder_invent();
	     
	} else {
	    prev->nobj = obj;           /* insert at end */
	    obj->nobj = 0;
	}

added:
	
	if (obj->otyp == LUCKSTONE
	    || (obj->oartifact && spec_ability(obj, SPFX_LUCK))) {
		/* new luckstone must be in inventory by this point
		 * for correct calculation */
		if (stone_luck(TRUE) >= 0) u.moreluck = LUCKADD;
		else u.moreluck = -LUCKADD;
	} else if (obj->otyp == OIL_LAMP || obj->otyp == MAGIC_LAMP)
		check_lamps();
	update_inventory();
	return(obj);
}

#endif /* OVL1 */
#ifdef OVLB

/* Add an item to the inventory unless we're fumbling, and give a message.
 * If there aren't any free inventory slots, we'll drop it instead.
 * If both success and failure messages are NULL, then we're just doing the
 * fumbling/slot-limit checking for a silent grab.
 * Note: will drop the whole bunch if the object merges first.
 */
struct obj *
hold_another_object(obj, drop_fmt, drop_arg, hold_msg)
struct obj *obj;
const char *drop_fmt, *drop_arg, *hold_msg;
{
	long oquan = obj->quan;
	if (!Blind) obj->dknown = 1;    /* maximize mergibility */
	if (Fumbling) {
		if (drop_fmt) pline(drop_fmt, drop_arg);
		dropy(obj);
	} else {
		obj = addinv(obj);
		if (inv_cnt() > 52
		    || ((obj->otyp != LOADSTONE || !obj->cursed)
			&& near_capacity() >= OVERLOADED)) {
			if (drop_fmt) pline(drop_fmt, drop_arg);
			dropx(obj);
		} else {
			if (hold_msg || drop_fmt) prinv(hold_msg, obj, oquan);
		}
	}
	return obj;
}

void
useup(obj)
register struct obj *obj;
{
	/*  Note:  This works correctly for containers because they */
	/*         (containers) don't merge.                        */
	if(obj->quan > 1L){
#ifndef NO_SIGNAL
		obj->in_use = FALSE;    /* no longer in use */
#endif
		obj->quan--;
		obj->owt = weight(obj);
	} else {
		setnotworn(obj);
		freeinv(obj);
		obfree(obj, (struct obj *) 0);  /* deletes contents also */
	}
}

#endif /* OVLB */
#ifdef OVL3

/* used by freeinv and doorganize to do list manipulation */
STATIC_OVL
void
unlinkinv(obj)
register struct obj *obj;
{
	register struct obj *otmp;

	if(obj == invent)
		invent = invent->nobj;
	else {
		for(otmp = invent; otmp->nobj != obj; otmp = otmp->nobj)
			if(!otmp->nobj) panic("unlinkinv");
		otmp->nobj = obj->nobj;
	}
	obj->nobj = 0;
}

void
freeinv(obj)
register struct obj *obj;
{
	unlinkinv(obj);

	if (obj->otyp == GOLD_PIECE) {
		u.ugold -= obj->quan;
		flags.botl = 1;
		return;
	} else if (obj->otyp == AMULET_OF_YENDOR) {
		if (!u.uhave.amulet) impossible ("don't have amulet?");
		u.uhave.amulet = 0;
	} else if (obj->otyp == CANDELABRUM_OF_INVOCATION) {
		if (!u.uhave.menorah) impossible ("don't have candelabrum?");
		u.uhave.menorah = 0;
	} else if (obj->otyp == BELL_OF_OPENING) {
		if (!u.uhave.bell) impossible ("don't have silver bell?");
		u.uhave.bell = 0;
	} else if (obj->otyp == SPE_BOOK_OF_THE_DEAD) {
		if (!u.uhave.book) impossible ("don't have the book?");
		u.uhave.book = 0;
#ifdef MULDGN
	} else if (is_quest_artifact(obj)) {
		if(!u.uhave.questart) impossible ("don't have the artifact?");
		u.uhave.questart = 0;
		set_artifact_intrinsic(obj, 0, W_ART);
#endif
	} else if (obj->oartifact) {
		set_artifact_intrinsic(obj, 0, W_ART);
	} else if (obj->otyp == OIL_LAMP || obj->otyp == MAGIC_LAMP
		   || obj->otyp == BRASS_LANTERN) {
		if (obj->lamplit) {
			obj->lamplit = 0;
			if (!Blind) pline("%s goes out!", The(xname(obj)));
		}
		check_lamps();
	} else if (obj->otyp == LOADSTONE) {
		curse(obj);
	} else if (obj->otyp == LUCKSTONE
		   || (obj->oartifact && spec_ability(obj, SPFX_LUCK))) {
		int luckbon = stone_luck(TRUE);
		if (!luckbon && !carrying(LUCKSTONE)) u.moreluck = 0;
		else if (luckbon >= 0) u.moreluck = LUCKADD;
		else u.moreluck = -LUCKADD;
		flags.botl = 1;
	}
	update_inventory();
}

void
delallobj(x, y)
int x, y;
{
	struct obj *otmp, *otmp2;

	for (otmp = level.objects[x][y]; otmp; otmp = otmp2) {
		if (otmp == uball)
			unpunish();
		/* after unpunish(), or might get deallocated chain */
		otmp2 = otmp->nexthere;
		if (otmp == uchain)
			continue;
		delobj(otmp);
	}
}

/* move objects from fobj/nexthere lists to buriedobjlist, keeping position
 * information */
void
bury_objs(x, y)
int x, y;
{
	struct obj *otmp, *otmp2;

	for (otmp = level.objects[x][y]; otmp; otmp = otmp2) {
		if (otmp == uball)
			unpunish();
		/* after unpunish(), or might get deallocated chain */
		otmp2 = otmp->nexthere;
		if (otmp == uchain)
			continue;
#ifdef WALKIES
		if (otmp->otyp == LEASH && otmp->leashmon != 0)
			o_unleash(otmp);
#endif
		if (otmp->lamplit) {
			otmp->lamplit = 0;
			check_lamps();
		}
		freeobj(otmp);
		if (otmp->otyp == ROCK) {
			/* melts into burying material */
			obfree(otmp, (struct obj *)0);
			continue;
		}
		otmp->nexthere = (struct obj *)0;
		otmp->nobj = level.buriedobjlist;
		level.buriedobjlist = otmp;
	}
	/* don't expect any engravings here, but just in case */
	del_engr_at(x, y);
	newsym(x, y);
}

/* delete buried object */
void
delburiedobj(obj)
struct obj *obj;
{
	struct obj *otmp;
	if (obj == level.buriedobjlist)
		level.buriedobjlist = obj->nobj;
	else {
		for (otmp = level.buriedobjlist; otmp; otmp = otmp->nobj)
			if (otmp->nobj == obj) {
				otmp->nobj = obj->nobj;
				break;
			}
		if (!otmp) panic("error in delburiedobj");
	}
	obfree(obj, (struct obj *) 0);  /* frees contents also */
}

/* move objects from buriedobjlist to fobj/nexthere lists */
void
unearth_objs(x, y)
int x, y;
{
	struct obj *otmp, *otmp2, *prevobj;

	remove_cadavers(&level.buriedobjlist);

	prevobj = (struct obj *) 0;
	for (otmp = level.buriedobjlist; otmp; otmp = otmp2) {
		otmp2 = otmp->nobj;
		if (otmp->ox == x && otmp->oy == y) {
			if (prevobj)
				prevobj->nobj = otmp2;
			else
				level.buriedobjlist = otmp2;
			if (is_organic(otmp) && rn2(2)) {
				/* rotted away */
				obfree(otmp, (struct obj *) 0);
			} else {
				otmp->nobj = fobj;
				fobj = otmp;
				place_object(otmp, x, y);
				stackobj(otmp);
			}
		} else
			prevobj = otmp;
	}
	del_engr_at(x, y);
	newsym(x, y);
}

#endif /* OVL3 */
#ifdef OVL2

/* destroy object in fobj chain (if unpaid, it remains on the bill) */
void
delobj(obj)
register struct obj *obj;
{
#ifdef WALKIES
	if(obj->otyp == LEASH && obj->leashmon != 0) o_unleash(obj);
#endif
	if (obj->otyp == AMULET_OF_YENDOR ||
			obj->otyp == CANDELABRUM_OF_INVOCATION ||
			obj->otyp == BELL_OF_OPENING ||
			obj->otyp == SPE_BOOK_OF_THE_DEAD) {
		/* player might be doing something stupid, but we
		 * can't guarantee that.  assume special artifacts
		 * are indestructible via drawbridges, and exploding
		 * chests, and golem creation, and ...
		 */
		return;
	}
	freeobj(obj);
	newsym(obj->ox,obj->oy);
	obfree(obj, (struct obj *) 0);  /* frees contents also */
}

/* unlink obj from chain starting with fobj */
void
freeobj(obj)
register struct obj *obj;
{
	register struct obj *otmp;

	if (obj == fobj)
	    fobj = fobj->nobj;
	else {
	    for(otmp = fobj; otmp; otmp = otmp->nobj)
		if (otmp->nobj == obj) {
		    otmp->nobj = obj->nobj;
		    break;
		}
	    if (!otmp) panic("error in freeobj");
	}
	remove_object(obj);
}

#endif /* OVL2 */
#ifdef OVL0

struct obj *
sobj_at(n,x,y)
register int n, x, y;
{
	register struct obj *otmp;

	for(otmp = level.objects[x][y]; otmp; otmp = otmp->nexthere)
		if(otmp->otyp == n)
		    return(otmp);
	return((struct obj *)0);
}

#endif /* OVL0 */
#ifdef OVLB

int
carried(obj)
register struct obj *obj;
{
	register struct obj *otmp;

	for(otmp = invent; otmp; otmp = otmp->nobj)
		if(otmp == obj) return(1);
	return(0);
}

struct obj *
carrying(type)
register int type;
{
	register struct obj *otmp;

	for(otmp = invent; otmp; otmp = otmp->nobj)
		if(otmp->otyp == type)
			return(otmp);
	return((struct obj *) 0);
}

boolean
have_lizard()
{
	register struct obj *otmp;

	for(otmp = invent; otmp; otmp = otmp->nobj)
		if(otmp->otyp == CORPSE && otmp->corpsenm == PM_LIZARD)
			return(TRUE);
	return(FALSE);
}

struct obj *
o_on(id, objchn)
unsigned int id;
register struct obj *objchn;
{
	struct obj *temp;

	while(objchn) {
		if(objchn->o_id == id) return(objchn);
		if (Has_contents(objchn) && (temp = o_on(id,objchn->cobj)))
			return temp;
		objchn = objchn->nobj;
	}
	return((struct obj *) 0);
}

boolean
obj_here(obj, x, y)
register struct obj *obj;
int x, y;
{
	register struct obj *otmp;

	for(otmp = level.objects[x][y]; otmp; otmp = otmp->nexthere)
		if(obj == otmp) return(TRUE);
	return(FALSE);
}

#endif /* OVLB */
#ifdef OVL2

struct obj *
g_at(x,y)
register int x, y;
{
	register struct obj *obj = level.objects[x][y];
	while(obj) {
	    if (obj->otyp == GOLD_PIECE) return obj;
	    obj = obj->nexthere;
	}
	return((struct obj *)0);
}

#endif /* OVL2 */
#ifdef OVLB

/* Make a gold object from the hero's gold. */
struct obj *
mkgoldobj(q)
register long q;
{
	register struct obj *otmp;

	otmp = mksobj(GOLD_PIECE, FALSE, FALSE);
	u.ugold -= q;
	otmp->quan = q;
	otmp->owt = weight(otmp);
	flags.botl = 1;
	return(otmp);
}

#endif /* OVLB */
#ifdef OVL1

STATIC_OVL void
compactify(buf)
register char *buf;
/* compact a string of inventory letters by dashing runs of letters */
{
	register int i1 = 1, i2 = 1;
	register char ilet, ilet1, ilet2;

	ilet2 = buf[0];
	ilet1 = buf[1];
	buf[++i2] = buf[++i1];
	ilet = buf[i1];
	while(ilet) {
		if(ilet == ilet1+1) {
			if(ilet1 == ilet2+1)
				buf[i2 - 1] = ilet1 = '-';
			else if(ilet2 == '-') {
				buf[i2 - 1] = ++ilet1;
				buf[i2] = buf[++i1];
				ilet = buf[i1];
				continue;
			}
		}
		ilet2 = ilet1;
		ilet1 = ilet;
		buf[++i2] = buf[++i1];
		ilet = buf[i1];
	}
}

/*
 * getobj returns:
 *      struct obj *xxx:        object to do something with.
 *      (struct obj *) 0        error return: no object.
 *      &zeroobj                explicitly no object (as in w-).
 */
struct obj *
getobj(let,word)
register const char *let,*word;
{
	register struct obj *otmp;
	register char ilet;
	char buf[BUFSZ], qbuf[QBUFSZ];
	char lets[BUFSZ];
	register int foo = 0;
	register char *bp = buf;
	xchar allowcnt = 0;     /* 0, 1 or 2 */
	boolean allowgold = FALSE, usegold = FALSE;
		/* Two possibilities: they can't use gold because it's illegal,
		 * or they can't use gold because they don't have any.
		 */
	boolean allowall = FALSE;
	boolean allownone = FALSE;
	xchar foox = 0;
	long cnt;
	boolean prezero = FALSE;

	if(*let == ALLOW_COUNT) let++, allowcnt = 1;
	if(*let == GOLD_CLASS) let++,
		usegold = TRUE, allowgold = (u.ugold ? TRUE : FALSE);
#ifdef POLYSELF
	/* Equivalent of an "ugly check" for gold */
	if (usegold && !strcmp(word, "eat") && !metallivorous(uasmon))
		usegold = allowgold = FALSE;
#endif
	if(*let == ALL_CLASSES) let++, allowall = TRUE;
	if(*let == ALLOW_NONE) let++, allownone = TRUE;
	/* "ugly check" for reading fortune cookies, part 1 */
	if(allowall && !strcmp(word, "read")) allowall = FALSE;

	if(allownone) *bp++ = '-';
	if(allowgold) *bp++ = def_oc_syms[GOLD_CLASS];
	if(bp > buf && bp[-1] == '-') *bp++ = ' ';

	ilet = 'a';
	for(otmp = invent; otmp; otmp = otmp->nobj){
	    if(!*let || index(let, otmp->oclass)) {
		bp[foo++] = flags.invlet_constant ? otmp->invlet : ilet;

		/* ugly check: remove inappropriate things */
		if((!strcmp(word, "take off") &&
		    (!(otmp->owornmask & (W_ARMOR | W_RING | W_AMUL | W_TOOL))
		     || (otmp==uarm && uarmc)
#ifdef TOURIST
		     || (otmp==uarmu && (uarm || uarmc))
#endif
		    ))
		|| (!strcmp(word, "wear") &&
		     (otmp->owornmask & (W_ARMOR | W_RING | W_AMUL | W_TOOL)))
							/* already worn */
		|| (!strcmp(word, "wield") &&
		    (otmp->owornmask & W_WEP))
		    ) {
			foo--;
			foox++;
		}

		/* Second ugly check; unlike the first it won't trigger an
		 * "else" in "you don't have anything else to ___".
		 */
		else if ((!strcmp(word, "wear") &&
		    (otmp->oclass == TOOL_CLASS &&
		     otmp->otyp != BLINDFOLD && otmp->otyp != TOWEL))
		|| (!strcmp(word, "wield") &&
		    (otmp->oclass == TOOL_CLASS &&
		     otmp->otyp != PICK_AXE && otmp->otyp != UNICORN_HORN))
#ifdef POLYSELF
		|| (!strcmp(word, "eat") && !is_edible(otmp))
#endif
		|| (!strcmp(word, "sacrifice") &&
		    (otmp->otyp != CORPSE &&
		     otmp->otyp != AMULET_OF_YENDOR &&
		     otmp->otyp != EYEBALL &&
		     otmp->otyp != SEVERED_HAND &&
		     otmp->otyp != FAKE_AMULET_OF_YENDOR))
		|| (!strcmp(word, "write with") &&
		    (otmp->oclass == TOOL_CLASS &&
		     otmp->otyp != MAGIC_MARKER && otmp->otyp != TOWEL))
		|| (!strcmp(word, "tin") &&
		    (otmp->otyp != CORPSE || !tinnable(otmp)))
		|| (!strcmp(word, "rub") &&
		    (otmp->oclass == TOOL_CLASS &&
		     otmp->otyp != OIL_LAMP && otmp->otyp != MAGIC_LAMP &&
		     otmp->otyp != BRASS_LANTERN))
		    )
			foo--;
	    } else {

		/* "ugly check" for reading fortune cookies, part 2 */
		if ((!strcmp(word, "read") && otmp->otyp == FORTUNE_COOKIE))
			allowall = TRUE;
	    }

	    if(ilet == 'z') ilet = 'A'; else ilet++;
	}
	bp[foo] = 0;
	if(foo == 0 && bp > buf && bp[-1] == ' ') *--bp = 0;
	Strcpy(lets, bp);       /* necessary since we destroy buf */
	if(foo > 5)                     /* compactify string */
		compactify(bp);

	if(!foo && !allowall && !allowgold && !allownone) {
		You("don't have anything %sto %s.",
			foox ? "else " : "", word);
		return((struct obj *)0);
	}
	for(;;) {
		cnt = 0;
		if (allowcnt == 2) allowcnt = 1;  /* abort previous count */
		if(!buf[0]) {
			Sprintf(qbuf, "What do you want to %s? [*]", word);
		} else {
			Sprintf(qbuf, "What do you want to %s? [%s or ?*]",
				word, buf);
		}
#ifdef REDO
		if (in_doagain)
		    ilet = readchar();
		else
#endif
		    ilet = yn_function(qbuf, NULL, '\0');
		if(ilet == '0') prezero = TRUE;
		while(digit(ilet) && allowcnt) {
#ifdef REDO
			if (ilet != '?' && ilet != '*') savech(ilet);
#endif
			cnt = 10*cnt + (ilet - '0');
			allowcnt = 2;   /* signal presence of cnt */
			ilet = readchar();
		}
		if(digit(ilet)) {
			pline("No count allowed with this command.");
			continue;
		}
		if(index(quitchars,ilet)) {
		    if(flags.verbose)
			pline("Never mind.");
		    return((struct obj *)0);
		}
		if(ilet == '-') {
			return(allownone ? &zeroobj : (struct obj *) 0);
		}
		if(ilet == def_oc_syms[GOLD_CLASS]) {
			if(!usegold){
				You("cannot %s gold.", word);
				return(struct obj *)0;
			} else if (!allowgold) {
				You("are not carrying any gold.");
				return(struct obj *)0;
			}
			if(cnt == 0 && prezero) return((struct obj *)0);
			/* Historic note: early Nethack had a bug which was
			 * first reported for Larn, where trying to drop 2^32-n
			 * gold pieces was allowed, and did interesting things
			 * to your money supply.  The LRS is the tax bureau
			 * from Larn.
			 */
			if(cnt < 0) {
	pline("The LRS would be very interested to know you have that much.");
				return(struct obj *)0;
			}

			if(!(allowcnt == 2 && cnt < u.ugold))
				cnt = u.ugold;
			return(mkgoldobj(cnt));
		}
		if(allowcnt == 2 && !strcmp(word,"throw")) {
			/* permit counts for throwing gold, but don't accept
			 * counts for other things since the throw code will
			 * split off a single item anyway */
			allowcnt = 1;
			if(cnt == 0 && prezero) return((struct obj *)0);
			if(cnt > 1) {
			    You("can only throw one item at a time.");
			    continue;
			}
		}
		if(ilet == '?' || ilet == '*') {
		    ilet = display_inventory(ilet == '?' ? lets : NULL, FALSE);
		    if(!ilet) continue;
		    if(ilet == '\033') {
			if(flags.verbose)
			    pline("Never mind.");
			return((struct obj *)0);
		    }
		    /* they typed a letter (not a space) at the prompt */
		}
#ifdef REDO
		savech(ilet);
#endif
		if(flags.invlet_constant) {
			for(otmp = invent; otmp; otmp = otmp->nobj)
				if(otmp->invlet == ilet) break;
		} else {
			if(ilet >= 'A' && ilet <= 'Z') ilet += 'z' - 'A' + 1;
			ilet -= 'a';
			for(otmp = invent; otmp && ilet;
					ilet--, otmp = otmp->nobj) ;
		}
		if(!otmp) {
			You("don't have that object.");
#ifdef REDO
			if (in_doagain) return((struct obj *) 0);
#endif
			continue;
		} else if (cnt < 0 || otmp->quan < cnt) {
			You("don't have that many!  You have only %ld.",
			    otmp->quan);
#ifdef REDO
			if (in_doagain) return((struct obj *) 0);
#endif
			continue;
		}
		break;
	}
	if(!allowall && let && !index(let,otmp->oclass)) {
		pline(silly_thing_to, word);
		return((struct obj *)0);
	}
	if(allowcnt == 2) {     /* cnt given */
		if(cnt == 0) return (struct obj *)0;
		if(cnt != otmp->quan) {
			register struct obj *obj = splitobj(otmp, cnt);
		/* Very ugly kludge necessary to prevent someone from trying
		 * to drop one of several loadstones and having the loadstone
		 * now be separate.
		 */
			if (!strcmp(word, "drop") &&
			    obj->otyp == LOADSTONE && obj->cursed)
				otmp->corpsenm = obj->invlet;
			if(otmp == uwep) setuwep(obj);
		}
	}
	return(otmp);
}

#endif /* OVL1 */
#ifdef OVLB

STATIC_PTR int
ckunpaid(otmp)
register struct obj *otmp;
{
	return((int)(otmp->unpaid));
}

static boolean
wearing_armor() {
	return((boolean)(uarm || uarmc || uarmf || uarmg || uarmh || uarms
#ifdef TOURIST
		|| uarmu
#endif
		));
}

static boolean
is_worn(otmp)
register struct obj *otmp;
{
    return((boolean)(!!(otmp->owornmask & (W_ARMOR | W_RING | W_AMUL | W_TOOL | W_WEP))));
}

static boolean
is_fully_identified(otmp)
register struct obj *otmp;
{
    return((boolean)(otmp->known && otmp->dknown && otmp->bknown && otmp->rknown
	   && objects[otmp->otyp].oc_name_known));
}

static NEARDATA const char removeables[] =
	{ ARMOR_CLASS, WEAPON_CLASS, RING_CLASS, AMULET_CLASS, TOOL_CLASS, 0 };

/* interactive version of getobj - used for Drop, Identify and */
/* Takeoff (A). Return the number of times fn was called successfully */
int
ggetobj(word, fn, mx)
register const char *word;
register int FDECL((*fn),(OBJ_P)), mx;
{
	char buf[BUFSZ], qbuf[QBUFSZ];
	register char *ip;
	register char sym;
	register int oletct = 0, iletct = 0;
	register boolean allflag = FALSE;
	char olets[20], ilets[20];
	int FDECL((*ckfn),(OBJ_P)) = (int (*)()) 0;
	xchar allowgold = (u.ugold && !strcmp(word, "drop")) ? 1 : 0; /* BAH */
	register boolean takeoff = !strcmp(word, "take off");
	register boolean ident = !strcmp(word, "identify");
	struct obj *obj;
	int unpaid, oc_of_sym;

	if(takeoff && !wearing_armor() && !uwep && !uamul &&
			!uleft && !uright && !ublindf) {
		You("are not wearing anything.");
		return(0);
	}
	if(!invent && !allowgold){
		You("have nothing to %s.", word);
		return(0);
	}

	if (allowgold) ilets[iletct++] = def_oc_syms[GOLD_CLASS];
	ilets[iletct] = '\0';   /* terminate for index() */
	unpaid = 0;
	for (obj = invent; obj; obj = obj->nobj) {
		sym = (char) def_oc_syms[(int) obj->oclass];
		if (!index(ilets, sym) && (!takeoff || is_worn(obj))
		    && (!ident || !is_fully_identified(obj))) {
			ilets[iletct++] = sym;
			/* necessary because of index() being used above */
			ilets[iletct] = '\0';
		}

		if (obj->unpaid) unpaid = 1;
	}

	if (ident && !iletct) {
	    You("have already identified all your possessions.");
	    return -1;          /* special case for seffects(read.c) */
	} else if (!takeoff && (unpaid || invent)) {
	    ilets[iletct++] = ' ';
	    if (unpaid) ilets[iletct++] = 'u';
	    if (invent) ilets[iletct++] = 'a';
	} else if (takeoff && invent) {
	    ilets[iletct++] = ' ';
	}
	ilets[iletct++] = 'i';
	ilets[iletct] = '\0';

	do {
	    Sprintf(qbuf,"What kinds of thing do you want to %s? [%s]",
		    word, ilets);
	    getlin(qbuf, buf);
	    if (buf[0] == '\033') return(0);
	    if (index(buf, 'i')) {
		(void) display_inventory(NULL, FALSE);
	    } else
		break;
	} while (1);

	ip = buf;
	olets[0] = 0;
	while ((sym = *ip++) != 0) {
		if(sym == ' ') continue;
		oc_of_sym = def_char_to_objclass(sym);
		if(takeoff && !(uwep && oc_of_sym == uwep->oclass)
		   && (oc_of_sym != MAXOCLASSES)) {
		    if(!index(removeables,oc_of_sym)) {
			pline("Not applicable.");
			return(0);
		    } else if(oc_of_sym == ARMOR_CLASS && !wearing_armor()) {
			You("are not wearing any armor.");
			return(0);
		    } else if(oc_of_sym == WEAPON_CLASS && !uwep) {
			You("are not wielding anything.");
			return(0);
		    } else if(oc_of_sym == RING_CLASS && !uright && !uleft) {
			You("are not wearing rings.");
			return(0);
		    } else if(oc_of_sym == AMULET_CLASS && !uamul) {
			You("are not wearing an amulet.");
			return(0);
		    } else if(oc_of_sym == TOOL_CLASS && !ublindf) {
			You("are not wearing a blindfold.");
			return(0);
		    }
		}
		if(oc_of_sym == GOLD_CLASS) {
			if(allowgold == 1)
				(*fn)(mkgoldobj(u.ugold));
			else if(!u.ugold)
				You("have no gold.");
			allowgold = 2;
		} else if(sym == 'a' || sym == 'A')
		    allflag = TRUE;
		else if(sym == 'u' || sym == 'U')
		    ckfn = ckunpaid;
		else if (oc_of_sym == MAXOCLASSES)
			You("don't have any %c's.", sym);
		else if (oc_of_sym != VENOM_CLASS) {/* venom doesn't show up */
			if (!index(olets, oc_of_sym)) {
				olets[oletct++] = oc_of_sym;
				olets[oletct] = 0;
			}
		}
	}
	if(allowgold == 2 && !oletct)
		return 1;       /* you dropped gold (or at least tried to) */
	else
		return askchain((struct obj **)&invent, olets, allflag,
				fn, ckfn, mx, word);
}

/*
 * Walk through the chain starting at objchn and ask for all objects
 * with olet in olets (if nonNULL) and satisfying ckfn (if nonNULL)
 * whether the action in question (i.e., fn) has to be performed.
 * If allflag then no questions are asked. Max gives the max nr of
 * objects to be treated. Return the number of objects treated.
 */
int
askchain(objchn, olets, allflag, fn, ckfn, mx, word)
struct obj **objchn;
register int allflag, mx;
register const char *olets, *word;      /* olets is an Obj Class char array */
register int FDECL((*fn),(OBJ_P)), FDECL((*ckfn),(OBJ_P));
{
	register struct obj *otmp, *otmp2;
	register char sym, ilet;
	register int cnt = 0, dud = 0, tmp;
	boolean takeoff, nodot, ident, ininv;
	char qbuf[QBUFSZ];

	takeoff = !strcmp(word, "take off");
	ident = !strcmp(word, "identify");
	nodot = (!strcmp(word, "nodot") || !strcmp(word, "drop") ||
		 ident || takeoff);
	ininv = (*objchn == invent);
	/* Changed so the askchain is interrogated in the order specified.
	 * For example, if a person specifies =/ then first all rings will be
	 * asked about followed by all wands -dgk
	 */
nextclass:
	ilet = 'a'-1;
	if (*objchn && (*objchn)->otyp == GOLD_PIECE) ilet--;   /* extra iteration */
	for (otmp = *objchn; otmp; otmp = otmp2) {
		if(ilet == 'z') ilet = 'A'; else ilet++;
		otmp2 = otmp->nobj;
		if (olets && *olets && otmp->oclass != *olets) continue;
		if (takeoff && !is_worn(otmp)) continue;
		if (ident && is_fully_identified(otmp)) continue;
		if (ckfn && !(*ckfn)(otmp)) continue;
		if (!allflag) {
			Strcpy(qbuf, ininv ?
				xprname(otmp, ilet, !nodot, 0L) : doname(otmp));
			Strcat(qbuf, "?");
			sym = (takeoff || ident || otmp->quan < 2L) ?
				nyaq(qbuf) : nyNaq(qbuf);
		}
		else    sym = 'y';

		if (sym == '#') {
		 /* Number was entered; split the object unless it corresponds
		    to 'none' or 'all'.  2 special cases: cursed loadstones and
		    welded weapons (eg, multiple daggers) will remain as merged
		    unit; done to avoid splitting an object that won't be
		    droppable (even if we're picking up rather than dropping).
		  */
		    if (!yn_number)
			sym = 'n';
		    else {
			sym = 'y';
			if (yn_number < otmp->quan && !welded(otmp) &&
			    (!otmp->cursed || otmp->otyp != LOADSTONE)) {
			    struct obj *otmpx = splitobj(otmp, yn_number);
			    if (!otmpx || otmpx->nobj != otmp2)
				impossible("bad object split in askchain");
			}
		    }
		}
		switch(sym){
		case 'a':
			allflag = 1;
		case 'y':
			tmp = (*fn)(otmp);
			if(tmp < 0) goto ret;
			cnt += tmp;
			if(--mx == 0) goto ret;
		case 'n':
			if(nodot) dud++;
		default:
			break;
		case 'q':
			/* special case for seffects() */
			if (ident) cnt = -1;
			goto ret;
		}
	}
	if (olets && *olets && *++olets)
		goto nextclass;
	if(!takeoff && (dud || cnt)) pline("That was all.");
	else if(!dud && !cnt) pline("No applicable objects.");
ret:
	return(cnt);
}

#endif /* OVLB */
#ifdef OVL2

STATIC_OVL char
obj_to_let(obj) /* should of course only be called for things in invent */
register struct obj *obj;
{
	register struct obj *otmp;
	register char ilet;

	if (obj->otyp == GOLD_PIECE)
		return GOLD_SYM;
	else if (flags.invlet_constant)
		return obj->invlet;
	ilet = 'a';
	for(otmp = invent; otmp && otmp != obj; otmp = otmp->nobj)
		if(++ilet > 'z') ilet = 'A';
	return((char)(otmp ? ilet : NOINVSYM));
}

/*
 * Print the indicated quantity of the given object.  If quan == 0L then use
 * the current quantity.
 */
void
prinv(prefix, obj, quan)
const char *prefix;
register struct obj *obj;
long quan;
{
#ifdef GCC_WARN
	long savequan = 0;
#else
	long savequan;
#endif
	if ( !prefix ) prefix = "";
	if (quan) {
		savequan = obj->quan;
		obj->quan = quan;
	}
	pline("%s%s%s",
	      prefix, *prefix ? " " : "",
	      xprname(obj, obj_to_let(obj), TRUE, 0L));
	if (quan) obj->quan = savequan;
}

#endif /* OVL2 */
#ifdef OVL1

char *
xprname(obj,let,dot,cost)
register struct obj *obj;
register char let;
register boolean dot;   /* append period; (dot && cost => Iu) */
register long cost;     /* cost (for inventory of unpaid or expended items) */
{
#ifdef LINT     /* handle static char li[BUFSZ]; */
	char li[BUFSZ];
#else
	static char li[BUFSZ];
#endif
	boolean use_invlet = flags.invlet_constant && let != CONTAINED_SYM;
    /*
     * If let is:
     *  *  Then obj == NULL and we are printing a total amount.
     *  >  Then the object is contained and doesn't have an inventory letter.
     */
    if (cost != 0 || let == '*') {
	/* if dot is true, we're doing Iu, otherwise Ix */
	Sprintf(li, "%c - %-45s %6ld zorkmid%s",
		(dot && use_invlet ? obj->invlet : let),
		(let != '*' ? doname(obj) : "Total:"), cost, plur(cost));
    } else if (obj->otyp == GOLD_PIECE) {
	Sprintf(li, "%ld gold piece%s%s", obj->quan, plur(obj->quan),
		(dot ? "." : ""));
    } else {
	/* ordinary inventory display or pickup message */
	Sprintf(li, "%c - %s",
		(use_invlet ? obj->invlet : let),
		doname(obj));
	if(dot) Strcat(li,".");
    }
    return li;
}

#endif /* OVL1 */
#ifdef OVLB

int
ddoinv()
{
	(void) display_inventory(NULL, FALSE);
	return 0;
}

/*
 *  find_unpaid()
 *
 *  Scan the given list of objects.  If last_found is NULL, return the first
 * /* IO.C
 * Functions for the drive, keyboard, and joystick.
 *
 */
/////////////////////////////////////////////////////////////////////////
void startcalibrate(void)
{
int n;
unsigned char validstick;
validstick = 0;
for (n=0;n<16;n++){
    joyread();
    xstick1[n] = xstick;
    if (xstick < 20000)validstick = 1;
    ystick1[n] = ystick;
    }
if (!validstick) {
    strcpy(showmsg, "No joystick connected");
    msgcnt = 20;
    return;
    }
firstin = 1;
minx = 0; miny = 0; maxx = 10000; maxy = 10000;
calibrating = 1;
joystick = 0;
}


void calibratestick(void)
{
int stickidx;
char s[80];
int cminx, cminy, cmaxx, cmaxy;
unsigned char valid;
unsigned short key;
unsigned xs, ys, n, m;
unsigned char button;
unsigned centerx, centery;
unsigned char validstick;
validstick = 0;
for (n=0;n<16;n++){
    joyread();
    xstick1[n] = xstick;
    if (xstick < 20000)validstick = 1;
    ystick1[n] = ystick;
    }
if (!validstick) {
    strcpy(showmsg, "No joystick connected");
    msgcnt = 20;
    return;
    }
for (n=0;n<16;n++){
    joyread();
    xstick1[n] = xstick;
    ystick1[n] = ystick;
    }
firstin = 1;
minx = 0; miny = 0; maxx = 10000; maxy = 10000;
joystick = 1;
waitnokey();
joystick = 0;
for(;;){
    black_status_screen();
    DrawControlPanel(36);
    printcenter("Move the joystick through its entire range.", smallfont, 180+CALIBRATEOFF);
    printcenter("Center stick, and press button 1 when done.", smallfont,  15 + 180+CALIBRATEOFF);
    printcenter("Press Esc to abort calibration.", smallfont, 30 + 180+CALIBRATEOFF);
    do {
	button = joyread();
	}while((xstick == 20000) || (ystick==20000));
    for (stickidx =0;stickidx < 16;stickidx++){
	xstick1[stickidx] = xstick;
	ystick1[stickidx] = ystick;}
    cminx = 32767;cminy=32767;
    cmaxx = 0; cmaxy = 0;
    for (xs=0,ys=0,m=0;m<16;m++){
	if (xstick1[m] < cminx) cminx = xstick1[m];
	if (xstick1[m] > cmaxx) cmaxx = xstick1[m];
	if (ystick1[m] < cminy) cminy = ystick1[m];
	if (ystick1[m] > cmaxy) cmaxy = ystick1[m];
	xs += xstick1[m];
	ys += ystick1[m]; }
    valid = 1;
    if ( (cmaxx > 50) &&  (cmaxx > (cminx<<1)) )valid = 0;
    if ( (cmaxx > 50) &&  (cmaxx > (cminx<<1)) )valid = 0;
    xs >>= 4;
    ys >>= 4;
    if (valid && firstin){
	minx = xs;miny = ys;
	maxx = xs;maxy = ys;
	firstin = 0;
	}
    if (valid){
	if (xs < minx)minx = xs;
	if (ys < miny)miny = ys;
	if (xs > maxx)maxx = xs;
	if (ys > maxy)maxy = ys;
	}
    sprintf(s,"Min Y:%-5d", miny);
    prints(s, smallfont, 320-50,  258+CALIBRATEOFF);
    sprintf(s,"Min X:%-5d", minx);
    prints(s, smallfont, 180-50,  278+CALIBRATEOFF);
    sprintf(s,"X:%-5d", xs);
    prints(s, smallfont, 250-50,  278+CALIBRATEOFF);
    sprintf(s,"Y:%-5d", ys);
    prints(s, smallfont, 414-50,  278+CALIBRATEOFF);
    sprintf(s,"Max X:%-5d", maxx);
    prints(s, smallfont, 454-50,  278+CALIBRATEOFF);
    sprintf(s,"Max Y:%-5d",maxy);
    prints(s, smallfont, 320-50,  298+CALIBRATEOFF);
    for(;;){
	key = readnextkey();
	if (!key)break;
	switch(key){
	    case 0x1c:
	    case 1:
		strcpy(showmsg, "Configuration aborted.");
		msgcnt = 50;
		return;
	    }
	}
    if (!(button & 0x10)){
	centerx = xs;centery = ys;
	rightthresh = (maxx - centerx)/3 + centerx;
	leftthresh  = centerx - (centerx-minx)/3;
	bottomthresh= (maxy - centery)/3 + centerx;
	topthresh   = centerx - (centery-miny)/3;
	if ( (centerx +20> maxx) || (centery +20> maxy) ||
	      (centerx < minx + 20) || (centery < miny + 20) ){
	    strcpy(showmsg, "Joystick error");
	    msgcnt = 20;
	    waitkey = 1;
	    return;
	    }
	joystick = 1;
	waitkey = 1;
	jv_playwavenote(35, 9, 32, 64);
	return;
	}
    update(scr);
    }
}

void loadframes(char *fliname)
{
unsigned char *todata, n;
unsigned line, size;
int fr;
float i;
int xoff, yoff, xsize, ysize, y;
unsigned short wormlevel;
flistart(fliname); // open fli file
font = fontload(screen);
fliframe();
smallfont = fontload(screen);
fliframe();
haircolor = screen[1];
bulletcolor = screen[2];
barcolor = screen[100];
equipcolor = screen[101];
selequipcolor = screen[102];
noequipcolor = screen[103];
fontshadowcolor    = screen[320+101];
fontcolor	   = screen[320+102];
for(n=0;n<NUMCOLORS;n++) Colors[n] = screen[320+103+n];
for (n=0;n<8;n++)starcolor[n] = screen[n*320];
for (n=0;n<10;n++)scopecolor[n] = screen[n*320 +4];
scopeoutline=screen[5];
scopeinside=screen[5+320];
dotcolor = screen[6];
for (n=0;n<16;n++)stargrad[n] = screen[7+n*320];
fliframe();
grabframesxy(dotframes, 2, 5, 5);

/*
boundsfind(screen, &xoff, &yoff, &xsize, &ysize); // Make smallest window.
todata = malloc(128000);
if (!todata){
    txtmode();
    printf("Out of memory\n");
    exit(0);
    }
*(unsigned short *)(&todata[0]) = xsize;
*(unsigned short *)(&todata[2]) = ysize;
*(unsigned short *)(&todata[4]) = xoff;
*(unsigned short *)(&todata[6]) = yoff;

for (size=8,line=xoff+yoff*320, y=0;y<ysize;line+=320,y++)
    size += packdata(&screen[line], &todata[size], xsize);
scopeframe = realloc(todata,size);
fliframe();
*/

grabframesxy(scopeframe, 1, 0, 0);
grabframes(stationframes, 7);
grabframes(radarframes, 49);
grabframes(planetframes, 7);
grabframes(rocketframes, 49);
grabframes(rockframes, 16);
grabframes(explodeframes, 19);
grabframes(bulletframes, 4);
grabframes(gbulletframes, 4);
grabframes(explode2frames, 19);
grabframes(mineframes, 42);
grabframes(mineexplode, 6);
grabframes(vortex, 36);
grabframes(planetframes1, 7);
grabframes(planetframes2, 7);
//grabframes(introframes, 2);
//grabframes(headframes, 1);
//grabframesxy(bolts2, 2, 0, 199);
//grabframes(frames, 50);
grabframesxy(energydsp, 1, 0, 0);
grabframesxy(ctrlpanel, 6*7, 0, 0);
grabframesxy(ctrlbuton, 4, 0, 0);
//grabframesxy(speaker, 6, 0, 0);
grabframesxy(antenna, 4, 0, 0);
grabframes(head2frames, 5);
grabframes(bobbleframes, 10);
grabframes(deadframes, 3);
grabframes(backgroundframes, 6);
grabframes(maskframes, 3);
grabframes(planetframes4, 7);
grabframes(bolts1, 4);
grabframes(bolts2, 4);
grabframes(newframes, 84);
closefli();
}
unsigned char readnextkey(void)
{
unsigned char key, key7f;
if  ((scancount & 0xff) != (readcount & 0xff) ){
    key = scancode[readcount++ & 0xff];
    key7f = key & 0x7f;
    if (key & 0x80){
	keytab[key7f] = 0;
	}
    else {
	keytab[key7f] = 1;
	keytime[key7f] = *jv_biostime;
	}
    return(key);
    }
return(0);
}

void flushkey(void)
{
unsigned char key;
for (;;){
    key = readnextkey();
    if (!key)break;
    }
}

unsigned char readjoykey(void)
{
static int ncount=0;
static unsigned lastbutton=0, lastselect=0;
static unsigned char oval = 0xff;
unsigned char key, button, go, val;
for (;;){
    key = readnextkey();
    if (!key)break;
    switch(key){
	case 0x4b:/* left */
	    return(1);
	case 0x4d:/* right */
	    return(2);
	case 0x48:/*updown*/
	    return(4);
	case 0x50:/*downdown*/
	    return(8);
	case 0x39:
	case 0x1c:
	    return(0x10);
	case 1:
	    return(0x20);
	case 0x14:  // t
	    return(0x40);
	case 0x21:  // f
	    return(0x80);
	case 0x22:  //g
	    return(0x44);
	case 0x2f:  //v
	    return(0x84);
	}
    if(key >= 0x02 && key <= 0x0d) return(key+0x84);
    if(key >= 0x3b && key <= 0x44) return(key+0x84);
    if(key == 0x57 || key == 0x58) return(key+0x84);
    }
if (joystick){
    val = 0;
    button = joyread();
    if (!(button & 0x10))val |= 0x10;
    if (!(button & 0x20))val |= 0x20;
    if (xstick < leftthresh)val |= 1;
    else if (xstick > rightthresh)val |= 2;
    if (ystick < topthresh) val |= 4;
    else if (ystick > bottomthresh) val |=8;
    if (val != oval) lastbutton = *jv_biostime;
    if (val) if ((!oval && (jv_timediff(lastselect) > 5) ) ||
	  (jv_timediff(lastbutton) > 5) ){
	oval = val;
	lastbutton = *jv_biostime;
	lastselect = *jv_biostime;
	return(val);
	}
    oval = val;
    }
return(0);
}


void waitnokey(void)
{
unsigned char button, nokey, key;
for (;;){
    nokey = 1;
    key = readnextkey();
    if (key)nokey = 0;
    if (keytab[0x39])nokey = 0;
    if (keytab[0x1c])nokey = 0;
    if (keytab[1])nokey = 0;
    if (joystick){
	button = joyread();
	if (!(button & 0x10))nokey = 0;
	if (!(button & 0x20))nokey = 0;
	}
    if (nokey)return;
    }
}
val = val;
	lastbutton = *jv_biostime;
	lastselect = *jv_biostime;
	return(val);
	}
    oval = val;
    }
return(0);
}


void waitnokey(void)
{
unsigned char button, nokey, key;
for (;;){
    nokey = 1;
    key = readnextkey();
    if (key)nokey = 0;
          P1  ƀ  dt t t t t    ,s   1."t 1Ҋ    @%        ]Ћ    @%        ]Ћ    @%   ^    s   E  N    X  @v  =   t	=   t  @%       1  H  %     1  H  %       } H    r:  r          1    %   =   t$1Ҋ      P1  ƀ     o   o^  tr^t  wr1wI  xg  y\      u      qr$qK  rC  k  X    rb  [r:[?  ]r]D  nR      @     r.vMrvV
th         1Ҋ  @      1Ҋ  @-  n    1Ҋ  @  S    1Ҋ  @M  8    1Ҋ  @m      @%   @|1  ƀ
  1  ƀ
       1Ҋ  @
      1Ҋ  @    1ۊ]1Ҋ    @%   [    @%   @|1  ƀ  1  ƀ   Y    @%   @|1  ƀM  1  ƀM   $      @%   }  1Ҋ  @m     1Ҋ    @%      ]Ћ    @%   #   ]Ћ    @%        E  N  w     @v  =   t  @%     LJ1  fǀ    1  ƀ-   1  ƀ-   1  ƀm   1  ƀ
          1Ҋ       o  1Ҋ   -    R  0P   0    1  f    1  H  %       =~   }    rG4  r	  <
  
  2	  
  rv   
  
  =O   u:j   @%   P
  	   1ۊ  3S  1  u=   t	    1Ҋ  1      {
    x tE=O   u:j   @%   P
  	   1ۊ  <S  1  C=O   u:j   @%   P
  	   1ۊ  3S  1    x t  1Ҋ     =   t     =y   u	=   t	=w   t  x u4     1ۊX1Ҋ     %   &o  %  =  u	    1Ҋ  1      #	  1    %   =   t$1Ҋ      P1  ƀ  =O   u:j   @%   P
  	   1ۊ  ES  1  g   o   o  tr^t  wr1w3  xs  y      u      qr$q  r  
    <  rb  [r:[5  ]r]Y  n  {  v  @  d  r6vtr   
   *  %       1  S}    1Ҋ  1  h        1  S    1Ҋ  1  .      1Ҋ    @%   1Ҋ    -  P  1Ҋ  1      o    1  S=    1Ҋ  1      5  1Ҋ    @%   1Ҋ    M  P  1Ҋ  1  L        @%   @|1  ƀ  1  ƀ     1Ҋ  1          1  S    1Ҋ  1      S    1  S    1Ҋ  1        1ۊ]1Ҋ    @%         @%   @|1  ƀ  1  ƀ         @%   @|1  ƀ=  1  ƀ=     1Ҋ  1      ^    1Ҋ  1      :    @%   }  1Ҋ  @]    1Ҋ  1  Y      1Ҋ    @%     1Ҋ  1        ]Ћ    @%       ]Ћ    @%   D    ]  U  N    1Ҋ  1  
    (    @v  =   t  @%       1Ҋ  1  F
         1Ҋ  1  "
       1  fǀ    1  ƀ   1  ƀ   1  ƀ     1Ҋ  1      1Ҋ     1  M  EE]1Ҋ         1Ҋ  1  f        1Ҋ   }  =O   u:j   @%   P
  	   1ۊ  NS  1  \  1Ҋ  1      t    1Ҋ     =O   u:j   @%   P
  	   1ۊ  WS  1    1Ҋ  1  n         0P   0    1  f  =O   u:j   @%   P
  	   1ۊ  `S  1  K  1Ҋ  1  
    f=O   u>1  P  @%   P  1Ɋ1ۊ  iS  1    1Ҋ  1  q
        rbA  r:  r
            rvM   x  k  =   u1Ҋ     %   
  =    x t  1Ҋ     =   uI  x t%  1ۊX1Ҋ     %   1Ҋ     %           1    %   =      1Ҋ      P1  ƀ  z@ {yxwvutsrqponed@&
 q   ʈ  I d  V  Ǌ    U    ۋ    U W e        .$
   1Ҋ  @}      1Ҋ  @    1  f   u61      1    R   	  Y    1  R    1Ҋ  @-  (    1Ҋ  @=  
    1Ҋ  @]    1Ҋ  @=  1    %   t81  1Ҋ=  1    %   1  M  1  1Ҋ  =  M  1  f  1  f   u31      1    R   	      @%   @|1  ƀ  1  ƀ   1    1    1    R0f	    1  f  1    R0f	  u  1ۊ]1Ҋ    @%   Q    @%   @|1  ƀ  1  ƀ       @%   @|1  ƀ=  1  ƀ=           @%   }  1Ҋ  @]     1Ҋ    @%   H   ]Ћ    @%   v   ]Ћ    @%     TR  @v  =   t	=   t  @%     1  1         1Ҋ   }              @%   ¡   %   1      }  1  1Ҋ  f}  f  1       t71  1Ҋ  }    1  }  1    =N   u   } 0} tEǀ      QE      EЈ    EЈ  E  &    EЈv  =N   tEǀ        EE]_^Y[SQVWU   EUUEE2EEE]_^Y[SQRVWU   =N   u	=5   tJ=   tAE    }	r
-EEE   tE  Ez]_^ZY[QVWU   EU]EE}rM}$  }r.}f  }r}w  }      }    }r}v@}   h  }^  0E%  PE $0%  P1PZ 7  Ex t10E%  PE $0%  PE@$0%  PQ[ "0E%  PE $0%  P1PZ    0E%  PE $0%  PE@$0%  Pd    0E%  PE $0%  PE@$0%  P_ q0E%  PE $0%  Ppb P0E%  PE $0%  Pd /0E%  PE $0%  PE@$0%  P*d ]_^YQVWU   EU]1ҊU1H  E=   }   =   t]1ҊU1E`E
EE} t]UE6'  =x   tUBEU,=y   tUBE藨  =   tUBE EEEEE]_^YSQRVWU   ;  E    }	|
EEE   t}	u5   ;  5  EE]_^ZY[SQRVWU    ]_^ZY[SQRVWU   /*      SCCS Id: @(#)pray.c     3.1     93/04/24        */
/* Copyright (c) Benson I. Margulies, Mike Stephenson, Steve Linhart, 1989. */
/* NetHack may be freely redistributed.  See license for details. */

#include "hack.h"
#include "epri.h"

STATIC_PTR int NDECL(prayer_done);
static int NDECL(in_trouble);
static void FDECL(fix_worst_trouble,(int));
static void FDECL(angrygods,(ALIGNTYP_P));
static void FDECL(pleased,(ALIGNTYP_P));
static void FDECL(godvoice,(ALIGNTYP_P,const char*));
static void FDECL(god_zaps_you,(ALIGNTYP_P));
static void FDECL(gods_angry,(ALIGNTYP_P));
static void FDECL(gods_upset,(ALIGNTYP_P));
static void FDECL(consume_offering,(struct obj *));
static boolean FDECL(water_prayer,(BOOLEAN_P));

/*
 * Logic behind deities and altars and such:
 * + prayers are made to your god if not on an altar, and to the altar's god
 *   if you are on an altar
 * + If possible, your god answers all prayers, which is why bad things happen
 *   if you try to pray on another god's altar
 * + sacrifices work basically the same way, but the other god may decide to
 *   accept your allegiance, after which they are your god.  If rejected,
 *   your god takes over with your punishment.
 * + if you're in Gehennom, all messages come from the chaotic god
 */
static
struct ghods {
	char    classlet;
	const char *law, *balance, *chaos;
}  gods[] = {
{'A', /* Central American */    "Quetzalcoatl", "Camaxtli", "Huhetotl"},
{'B', /* Hyborian */            "Mitra", "Crom", "Set"},
{'C', /* Babylonian */          "Anu", "Ishtar", "Anshar"},
{'D', /* Babylonian */          "Anu", "Ishtar", "Anshar"},
{'E', /* Elven */               "Solonor Thelandira",
					"Aerdrie Faenya", "Erevan Ilesere"},
{'F', /* Special */             "Earth", "Fire", "Ash"},
{'H', /* Greek */               "Athena", "Hermes", "Poseidon"},
{'I', /* Special */             "Air", "Frost", "Smoke"},
{'K', /* Celtic */              "Lugh", "Brigit", "Macannan Mac Lir"},
{'L', /* Special */             "Eluvian", "Moon", "Lycanthus"},
{'M', /* Chinese */             "Shan Lai Ching", "Chih Sung-tzu", "Huan Ti"},
{'P', /* Chinese */             "Shan Lai Ching", "Chih Sung-tzu", "Huan Ti"},
{'R', /* Nehwon */              "Issek", "Mog", "Kos"},
{'S', /* Japanese */            "Amaterasu Omikami", "Raiden", "Susanowo"},
#ifdef TOURIST
{'T', /* Discworld */           "Blind Io", "The Lady", "Offler"},
#endif
{'U', /* Egyptian */            "Seeker", "Osiris", "Seth"},
{'V', /* Norse */               "Tyr", "Odin", "Loki"},
{'W', /* Egyptian */            "Ptah", "Thoth", "Anhur"},
{0,0,0,0}
};

/*
 *      Moloch, who dwells in Gehennom, is the "renegade" cruel god
 *      responsible for the theft of the Amulet from Marduk, the Creator.
 *      Moloch is unaligned.
 */
static const char       *Moloch = "Moloch";

static const char *godvoices[] = {
    "booms out",
    "thunders",
    "rings out",
    "booms",
};

/* values calculated when prayer starts, and used when completed */
static aligntyp p_aligntyp;
static int p_trouble;
static int p_type; /* (-1)-3: (-1)=really naughty, 3=really good */

#define PIOUS 20
#define DEVOUT 14
#define FERVENT 9
#define STRIDENT 4

#define TROUBLE_STONED 11
#define TROUBLE_SLIMED 10
#define TROUBLE_STRANGLED 9
#define TROUBLE_LAVA 8
#define TROUBLE_SICK 7
#define TROUBLE_STARVING 6
#define TROUBLE_HIT 5
#define TROUBLE_LYCANTHROPE 4
#define TROUBLE_STUCK_IN_WALL 3
#define TROUBLE_CURSED_BLINDFOLD 2
#define TROUBLE_CURSED_LEVITATION 1

#define TROUBLE_PUNISHED (-1)
#define TROUBLE_CURSED_ITEMS (-2)
#define TROUBLE_BLIND (-3)
#define TROUBLE_HUNGRY (-4)
#define TROUBLE_POISONED (-5)
#define TROUBLE_WOUNDED_LEGS (-6)
#define TROUBLE_STUNNED (-7)
#define TROUBLE_CONFUSED (-8)
#define TROUBLE_HALLUCINATION (-9)

/* We could force rehumanize of polyselfed people, but we can't tell
   unintentional shape changes from the other kind. Oh well. */

/* Return 0 if nothing particular seems wrong, positive numbers for
   serious trouble, and negative numbers for comparative annoyances. This
   returns the worst problem. There may be others, and the gods may fix
   more than one.

This could get as bizarre as noting surrounding opponents, (or hostile dogs),
but that's really hard.
 */

#define ugod_is_angry() (u.ualign.record < 0)
#define on_altar()      IS_ALTAR(levl[u.ux][u.uy].typ)
#define on_shrine()     ((levl[u.ux][u.uy].altarmask & AM_SHRINE) != 0)
#define a_align(x,y)    ((aligntyp)Amask2align(levl[x][y].altarmask & ~AM_SHRINE))

static int
in_trouble()
{
	register struct obj *otmp;
	int i, j, count=0;

/* Borrowed from eat.c */

#define SATIATED        0
#define NOT_HUNGRY      1
#define HUNGRY          2
#define WEAK            3
#define FAINTING        4
#define FAINTED         5
#define STARVED         6

	if(Stoned) return(TROUBLE_STONED);
	if(Slimed) return(TROUBLE_SLIMED);
	if(Strangled) return(TROUBLE_STRANGLED);
	if(u.utrap && u.utraptype == TT_LAVA) return(TROUBLE_LAVA);
	if(Sick) return(TROUBLE_SICK);
	if(u.uhs >= WEAK) return(TROUBLE_STARVING);
	if(u.uhp < 5 || (u.uhp*7 < u.uhpmax)) return(TROUBLE_HIT);
#ifdef POLYSELF
	if(u.ulycn >= 0 && !pl_character[0] == 'L') return(TROUBLE_LYCANTHROPE);
#endif
	for (i= -1; i<=1; i++) for(j= -1; j<=1; j++) {
		if (!i && !j) continue;
		if (!isok(u.ux+i, u.uy+j) || IS_ROCK(levl[u.ux+i][u.uy+j].typ))
			count++;
	}
	if(count==8
#ifdef POLYSELF
	    && !p